/*
 * @(#)RERView.java  1.0  2006-02-07
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.rer;

import ch.hslu.cm.app.action.Arrangeable;
import ch.hslu.cm.rer.diagram.*;
import ch.hslu.cm.rer.model.*;
import ch.randelshofer.io.*;
import ch.randelshofer.util.ZipFiles;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.beans.*;
import java.io.*;
import java.text.*;
import java.util.*;
import java.util.prefs.*;
import java.util.zip.*;
import javax.swing.*;
import javax.swing.border.*;
import java.lang.reflect.*;
import java.net.URI;
import org.jhotdraw.app.*;
import org.jhotdraw.app.action.ActionUtil;
import org.jhotdraw.app.action.edit.CopyAction;
import org.jhotdraw.app.action.edit.CutAction;
import org.jhotdraw.app.action.edit.DuplicateAction;
import org.jhotdraw.app.action.edit.PasteAction;
import org.jhotdraw.app.action.edit.RedoAction;
import org.jhotdraw.app.action.edit.SelectAllAction;
import org.jhotdraw.app.action.edit.UndoAction;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.action.BringToFrontAction;
import org.jhotdraw.draw.action.SendToBackAction;
import org.jhotdraw.draw.event.ToolAdapter;
import org.jhotdraw.draw.event.ToolEvent;
import org.jhotdraw.draw.event.ToolListener;
import org.jhotdraw.draw.io.DOMStorableInputOutputFormat;
import org.jhotdraw.draw.io.ImageOutputFormat;
import org.jhotdraw.draw.io.InputFormat;
import org.jhotdraw.draw.io.OutputFormat;
import org.jhotdraw.draw.tool.ConnectionTool;
import org.jhotdraw.draw.tool.CreationTool;
import org.jhotdraw.draw.tool.DelegationSelectionTool;
import org.jhotdraw.draw.tool.TextAreaCreationTool;
import org.jhotdraw.draw.tool.Tool;
import org.jhotdraw.gui.URIChooser;
import org.jhotdraw.gui.filechooser.ExtensionFileFilter;
import org.jhotdraw.undo.*;
import org.jhotdraw.util.*;
import org.jhotdraw.xml.*;

/**
 * RERView.
 *
 *
 * @author Werner Randelshofer
 * @version 1.0 2006-02-07 Created.
 */
public class RERView extends AbstractView
        implements Arrangeable {

    private Arrangeable.Arrangement arrangement;
    private JSplitPane splitPane;
    private JTabbedPane splittedProblemPanel, splittedSolutionPanel;
    private org.jhotdraw.util.ResourceBundleUtil labels;
    private Preferences prefs;
    private UndoRedoManager undo;
    private DrawingEditor editor;
    private RERFactory domFactory;
    private boolean isOptionsVisible = true;
    private GridConstrainer visibleConstrainer = new GridConstrainer(10, 10);
    private GridConstrainer invisibleConstrainer = new GridConstrainer(1, 1);

    private class ToolButtonListener implements ItemListener {

        private Tool tool;

        public ToolButtonListener(Tool t) {
            this.tool = t;
        }

        @Override
        public void itemStateChanged(ItemEvent evt) {
            if (evt.getStateChange() == ItemEvent.SELECTED) {
                editor.setTool(tool);
            }
        }
    }

    /** Creates new form. */
    public RERView() {
    }

    protected RelationalERDiagram createDrawing() {
        RelationalERDiagram d = new RelationalERDiagram();
        configureDrawing(d);
        return d;
    }

    protected void configureDrawing(RelationalERDiagram d) {
        DOMStorableInputOutputFormat domIOF = new DOMStorableInputOutputFormat(new RERFactory());
        LinkedList<InputFormat> inputFormats = new LinkedList<InputFormat>();
        inputFormats.add(domIOF);
        LinkedList<OutputFormat> outputFormats = new LinkedList<OutputFormat>();
        outputFormats.add(domIOF);
        outputFormats.add(new ImageOutputFormat("PNG", "Portable Network Graphics", "png", BufferedImage.TYPE_INT_RGB));
        d.setInputFormats(inputFormats);
        d.setOutputFormats(outputFormats);
    }

    @Override
    public void init() {
        super.init();
        labels = org.jhotdraw.util.ResourceBundleUtil.getBundle("ch.hslu.cm.rer.Labels");
        prefs = Preferences.userNodeForPackage(getClass());
        undo = new UndoRedoManager();

        initComponents();
        initActions();

        problemDrawingView.setDrawing(createDrawing());
        solutionDrawingView.setDrawing(createDrawing());

        problemDrawingView.setEmptyDrawingMessage(labels.getString("emptyProblemMessage"));
        solutionDrawingView.setEmptyDrawingMessage(labels.getString("emptySolutionMessage"));

        splitPane = new JSplitPane();
        splitPane.setBorder(new EmptyBorder(0, 0, 0, 0));
        splitPane.add(splittedProblemPanel = new JTabbedPane(), JSplitPane.LEFT);
        splitPane.add(splittedSolutionPanel = new JTabbedPane(), JSplitPane.RIGHT);
        splittedProblemPanel.setTabPlacement(JTabbedPane.BOTTOM);
        splittedSolutionPanel.setTabPlacement(JTabbedPane.BOTTOM);

        // Get/Store divider location from/to preferences
        splitPane.setDividerLocation(prefs.getInt("project.dividerLocation", 100));
        splitPane.addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (evt.getPropertyName().equals("dividerLocation")) {
                    prefs.putInt("project.dividerLocation", splitPane.getDividerLocation());
                }
            }
        });
        mainSplitPane.setDividerLocation(prefs.getInt("project.mainDividerLocation", 400));
        mainSplitPane.addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (evt.getPropertyName().equals("dividerLocation")) {
                    prefs.putInt("project.mainDividerLocation", mainSplitPane.getDividerLocation());
                }
            }
        });


        // Get arrangement from preferences
        setArrangement(Arrangeable.Arrangement.valueOf(
                prefs.get("project.arrangement", Arrangeable.Arrangement.CASCADE.toString())));

        setDrawingEditor(new DefaultDrawingEditor());

        setOptionsVisible(prefs.getBoolean("project.optionsVisible", true));

        splittedProblemPanel.setTabLayoutPolicy(JTabbedPane.WRAP_TAB_LAYOUT);
        splittedSolutionPanel.setTabLayoutPolicy(JTabbedPane.WRAP_TAB_LAYOUT);
        tabbedPane.setTabLayoutPolicy(JTabbedPane.WRAP_TAB_LAYOUT);
        mainSplitPane.putClientProperty("Quaqua.SplitPane.style", "bar");
        splitPane.putClientProperty("Quaqua.SplitPane.style", "bar");
        Font smallSystemFont = UIManager.getFont("SmallSystemFont");
        if (smallSystemFont != null) {
            splittedProblemPanel.setFont(smallSystemFont);
            splittedSolutionPanel.setFont(smallSystemFont);
            tabbedPane.setFont(smallSystemFont);
        }
        splittedProblemPanel.putClientProperty("Quaqua.Component.visualMargin", new Insets(0, 0, 3, 0));
        splittedSolutionPanel.putClientProperty("Quaqua.Component.visualMargin", new Insets(0, 0, 3, 0));
        tabbedPane.putClientProperty("Quaqua.Component.visualMargin", new Insets(0, 0, 3, 0));
        problemScrollPane.putClientProperty("Quaqua.TabbedPaneChild.contentInsets", new Insets(0, 0, 0, 0));
        solutionScrollPane.putClientProperty("Quaqua.TabbedPaneChild.contentInsets", new Insets(0, 0, 0, 0));

        undo.addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                setHasUnsavedChanges(undo.hasSignificantEdits());
            }
        });

        problemDrawingView.getDrawing().addUndoableEditListener(undo);
        solutionDrawingView.getDrawing().addUndoableEditListener(undo);
        optionsPanel.addUndoableEditListener(undo);
    }

    private void initActions() {
        ActionMap am = getActionMap();
        am.put(UndoAction.ID, undo.getUndoAction());
        am.put(RedoAction.ID, undo.getRedoAction());
        am.put(SelectAllAction.ID, new SelectAllAction());
        am.put(DuplicateAction.ID, new DuplicateAction());

        //   am.put(GroupAction.ID, new GroupAction(editor));
        //   am.put(UngroupAction.ID, new UngroupAction(editor));
        AbstractAction toggleOptionsAction = new AbstractAction(labels.getString("workAssignment")) {

            @Override
            public void actionPerformed(ActionEvent evt) {
                setOptionsVisible(!isOptionsVisible());
                putValue(ActionUtil.SELECTED_KEY, isOptionsVisible());
            }
        };
        toggleOptionsAction.putValue(ActionUtil.SELECTED_KEY, isOptionsVisible());
        am.put("toggleOptionsPanel", toggleOptionsAction);
    }

    public RelationalERModel getProblemModel() {
        return ((RelationalERDiagram) problemDrawingView.getDrawing()).getERModel();
    }

    public RelationalERModel getSolutionModel() {
        return ((RelationalERDiagram) solutionDrawingView.getDrawing()).getERModel();
    }

    @Override
    public void write(URI f, URIChooser u) throws IOException {
        PrintWriter out = null;
        try {
            out = new PrintWriter(new OutputStreamWriter(new BufferedOutputStream(new FileOutputStream(new File(f))), "UTF8"));
            NanoXMLDOMOutput domo = new NanoXMLDOMOutput(new RERFactory());
            domo.openElement("ConceptModelerRER");
            domo.openElement("Chapter");
            domo.writeObject(optionsPanel.getChapterText());
            domo.closeElement();
            domo.openElement("Title");
            domo.writeObject(optionsPanel.getTitleText());
            domo.closeElement();
            domo.openElement("Instructions");
            domo.writeObject(optionsPanel.getInstructionsTextHTML());
            domo.closeElement();
            domo.openElement("Language");
            domo.writeObject(optionsPanel.getInstructionLanguage().getLanguage());
            domo.closeElement();
            domo.openElement("Problem");
            RelationalERDiagram diagram;
            diagram = (RelationalERDiagram) problemDrawingView.getDrawing();
            domo.writeObject(diagram.getSimulation());
            domo.writeObject(diagram);
            domo.closeElement();
            domo.openElement("Solution");
            diagram = (RelationalERDiagram) solutionDrawingView.getDrawing();
            domo.writeObject(diagram.getSimulation());
            domo.writeObject(diagram);
            domo.closeElement();
            domo.closeElement();
            domo.print(out);
        } finally {
            out.close();
            //if (out != null) try { out.close(); } catch (IOException e) {};
        }

    }

    @Override
    public void read(URI f, URIChooser c) throws IOException {
        InputStream in = null;
        try {
            in = new BufferedInputStream(new FileInputStream(new File(f)));
            NanoXMLDOMInput domi = new NanoXMLDOMInput(new RERFactory(), in);
            domi.openElement("ConceptModelerRER");
            final String chapterText;
            if (domi.getElementCount("Chapter") > 0) {
                domi.openElement("Chapter");
                chapterText = (String) domi.readObject();
                domi.closeElement();
            } else {
                chapterText = "";
            }
            final String titleText;
            if (domi.getElementCount("Title") > 0) {
                domi.openElement("Title");
                titleText = (String) domi.readObject();
                domi.closeElement();
            } else {
                titleText = "";
            }
            final String instructionsText;
            if (domi.getElementCount("Instructions") > 0) {
                domi.openElement("Instructions");
                instructionsText = (String) domi.readObject();

                domi.closeElement();
            } else {
                instructionsText = "";
            }
            final String language;
            if (domi.getElementCount("Language") > 0) {
                domi.openElement("Language");
                language = (String) domi.readObject();
                domi.closeElement();
            } else {
                language = "en";
            }
            final RelationalERDiagram pDiagram;
            if (domi.getElementCount("Problem") > 0) {
                domi.openElement("Problem");
                RelationalERModel pSimulation = (RelationalERModel) domi.readObject(0);
                pDiagram = (RelationalERDiagram) domi.readObject(1);
                configureDrawing(pDiagram);
                domi.closeElement();
            } else {
                pDiagram = createDrawing();
            }
            final RelationalERDiagram sDiagram;
            if (domi.getElementCount("Solution") > 0) {
                domi.openElement("Solution");
                RelationalERModel sSimulation = (RelationalERModel) domi.readObject(0);
                sDiagram = (RelationalERDiagram) domi.readObject(1);
                configureDrawing(sDiagram);
                domi.closeElement();
            } else {
                sDiagram = createDrawing();
            }
            SwingUtilities.invokeAndWait(new Runnable() {

                @Override
                public void run() {
                    problemDrawingView.getDrawing().removeUndoableEditListener(undo);
                    solutionDrawingView.getDrawing().removeUndoableEditListener(undo);
                    problemDrawingView.setDrawing(pDiagram);
                    solutionDrawingView.setDrawing(sDiagram);
                    problemDrawingView.getDrawing().addUndoableEditListener(undo);
                    solutionDrawingView.getDrawing().addUndoableEditListener(undo);
                    optionsPanel.setChapterText(chapterText);
                    optionsPanel.setTitleText(titleText);
                    optionsPanel.setInstructionsText(instructionsText);
                    optionsPanel.setInstructionLanguage(new Locale(language));
                    undo.discardAllEdits();
                }
            });
        } catch (InterruptedException e) {
            InternalError error = new InternalError();
            e.initCause(e);
            throw error;
        } catch (InvocationTargetException e) {
            InternalError error = new InternalError();
            e.initCause(e);
            throw error;
        } finally {
            if (in != null) {
                try {
                    in.close();
                } catch (IOException e) {
                }
            }
            ;
        }
    }

    /**
     * By convention, this method is never invoked on the AWT event dispatcher
     * thread.
     */
    public void export(URI u, javax.swing.filechooser.FileFilter filter, Component accessory) throws IOException {
        File f = new File(u);
        prefs.put("projectExportFile", f.getPath());

        if (filter instanceof ExtensionFileFilter) {
            ExtensionFileFilter eff = (ExtensionFileFilter) filter;
            if (eff.getExtensions().contains("html")) {
                String baseName = f.getName();
                File baseDir = f.getParentFile();

                if (baseName.toLowerCase().endsWith(".html")) {
                    baseName = baseName.substring(0, baseName.length() - 5);
                } else if (baseName.toLowerCase().endsWith(".htm")) {
                    baseName = baseName.substring(0, baseName.length() - 4);
                }

                // Write libraries
                ZipInputStream zin = null;
                try {
                    zin = new ZipInputStream(getClass().getResourceAsStream("/htmlexport.zip"));
                    ZipFiles.unzip(zin, baseDir);
                } finally {
                    if (zin != null) {
                        zin.close();
                    }
                }

                // Write HTML page
                InputStream in = null;
                String html = null;
                try {
                    in = getClass().getResourceAsStream(
                            (problemDrawingView.getDrawing().getChildCount() == 0
                            && solutionDrawingView.getDrawing().getChildCount() == 0)
                            ? "/ch/hslu/cm/prototype_cmtext_sco.html"
                            : "/ch/hslu/cm/prototype_cmapplet_sco.html");
                    String htmlPrototype = Streams.toString(in);
                    String titleTextHTML = optionsPanel.getTitleText();
                    if (titleTextHTML == null || titleTextHTML.trim().length() == 0) {
                        titleTextHTML = "";
                    } else {
                        titleTextHTML = "<span class=\"pageTitle\"> &middot; <!-- InstanceBeginEditable name=\"pageRegion\" -->"
                                + titleTextHTML
                                + "<!-- InstanceEndEditable --></span></div>";
                    }
                    html = MessageFormat.format(
                            htmlPrototype,
                            optionsPanel.getChapterText(),
                            titleTextHTML.length() == 0 ? optionsPanel.getChapterText() : optionsPanel.getTitleText(),
                            titleTextHTML,
                            optionsPanel.getInstructionsTextBodyContent(),
                            optionsPanel.getInstructionLanguage().getLanguage(),
                            baseName + ".xml",
                            "ch.hslu.cm.rer.RERLiveConnectApplet",
                            "cmrerapplet.jar");
                } finally {
                    if (in != null) {
                        in.close();
                    }
                }
                OutputStreamWriter w = null;
                try {
                    w = new OutputStreamWriter(new FileOutputStream(new File(baseDir, baseName + ".html")), "UTF-8");
                    w.write(html);
                } finally {
                    if (w != null) {
                        w.close();
                    }
                }

                // Write project document
                write(new File(baseDir, baseName + ".xml").toURI(),null);

                return;
            }
        }
        throw new IOException(labels.getFormatted("exportCantExportFileAs", f, filter.getDescription()));
    }

    public void setDrawingEditor(DrawingEditor newValue) {
        if (editor != null) {
            editor.remove(problemDrawingView);
            editor.remove(solutionDrawingView);
        }
        editor = newValue;
        if (editor != null) {
            editor.add(problemDrawingView);
            editor.add(solutionDrawingView);
        }
    }

    public DrawingEditor getDrawingEditor() {
        return editor;
    }

    public void setOptionsVisible(boolean newValue) {
        if (isOptionsVisible != newValue) {
            boolean oldValue = isOptionsVisible;
            isOptionsVisible = newValue;

            removeAll();
            if (isOptionsVisible) {
                add(mainSplitPane);
                mainRightPanel.add(mainPanel);
                optionsPanel.invalidate();
                mainLeftPanel.invalidate();
                mainRightPanel.invalidate();

            } else {
                add(mainPanel);
            }
            revalidate();
            repaint();

            firePropertyChange("optionsVisible", oldValue, newValue);
            prefs.putBoolean("project.optionsVisible", newValue);
        }
    }

    public boolean isOptionsVisible() {
        return isOptionsVisible;
    }

    public void setGridVisible(boolean newValue) {
        boolean oldValue = isGridVisible();
        problemDrawingView.setConstrainerVisible(newValue);
        solutionDrawingView.setConstrainerVisible(newValue);
        firePropertyChange("gridVisible", oldValue, newValue);
        prefs.putBoolean("project.gridVisible", newValue);
    }

    public boolean isGridVisible() {
        return problemDrawingView.getConstrainer() == visibleConstrainer;
    }

    public void setScaleFactor(double newValue) {
        double oldValue = getScaleFactor();
        problemDrawingView.setScaleFactor(newValue);
        solutionDrawingView.setScaleFactor(newValue);

        firePropertyChange("scaleFactor", oldValue, newValue);
        prefs.putDouble("project.scaleFactor", newValue);
    }

    public double getScaleFactor() {
        return problemDrawingView.getScaleFactor();
    }

    @Override
    public void clear() {
        RelationalERDiagram problemDiagram = createDrawing();
        problemDiagram.setSimulation(new RelationalERModel());
        RelationalERDiagram solutionDiagram = createDrawing();
        solutionDiagram.setSimulation(new RelationalERModel());
        optionsPanel.clear();
        problemDrawingView.setDrawing(problemDiagram);
        solutionDrawingView.setDrawing(solutionDiagram);
    }

    @Override
    public void setArrangement(Arrangeable.Arrangement newValue) {
        if (newValue != arrangement) {
            Arrangeable.Arrangement oldValue = arrangement;
            arrangement = newValue;

            mainPanel.removeAll();

            switch (arrangement) {
                case HORIZONTAL:
                    splittedProblemPanel.add(problemScrollPane, labels.getString("problem"));
                    splittedSolutionPanel.add(solutionScrollPane, labels.getString("solution"));
                    splitPane.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
                    mainPanel.add(splitPane);
                    break;
                case VERTICAL:
                    splittedProblemPanel.add(problemScrollPane, labels.getString("problem"));
                    splittedSolutionPanel.add(solutionScrollPane, labels.getString("solution"));
                    splitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
                    mainPanel.add(splitPane);
                    break;
                case CASCADE:
                    tabbedPane.add(problemScrollPane, labels.getString("problem"));
                    tabbedPane.add(solutionScrollPane, labels.getString("solution"));
                    mainPanel.add(tabbedPane);
                    break;
            }
            mainPanel.validate();
            mainPanel.repaint();

            prefs.put("project.arrangement", newValue.toString());
            firePropertyChange("arrangement", oldValue, newValue);
        }
    }

    @Override
    public Arrangeable.Arrangement getArrangement() {
        return arrangement;
    }

    public Collection<Action> getDrawingActions() {
        ActionMap am = getActionMap();
        LinkedList<Action> a = new LinkedList<Action>();
        a.add(am.get(CutAction.ID));
        a.add(am.get(CopyAction.ID));
        a.add(am.get(PasteAction.ID));
        a.add(new SelectAllAction());

        return a;
    }

    public Collection<Action> getSelectionActions() {
        ActionMap am = getActionMap();
        LinkedList<Action> a = new LinkedList<Action>();
        a.add(am.get(DuplicateAction.ID));
        /*
        a.add(null); // separator
        a.add(am.get(GroupAction.ID));
        a.add(am.get(UngroupAction.ID);
         */
        a.add(null); // separator
        a.add(new BringToFrontAction(editor));
        a.add(new SendToBackAction(editor));

        return a;
    }

    public void addDrawActionsTo(JToolBar tb) {
        ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.rer.Labels");
        ResourceBundleUtil drawLabels = ResourceBundleUtil.getBundle("org.jhotdraw.draw.Labels");

        JToggleButton t;
        Tool tool;
        ButtonGroup group;
        HashMap<AttributeKey, Object> attributes;
        RelationshipFigure rf;

        group = new ButtonGroup();
        // Selection tool
        Tool selectionTool = new DelegationSelectionTool(getDrawingActions(), getSelectionActions());
        editor.setTool(selectionTool);
        t = new JToggleButton();
        ToolHandler toolHandler = new ToolHandler(t);
        drawLabels.configureToolBarButton(t, "selectionTool");
        t.setSelected(true);
        t.addItemListener(
                new ToolButtonListener(selectionTool));
        t.setFocusable(false);
        group.add(t);
        tb.add(t);
        tb.addSeparator();

        // AttributeKeys for the entities
        attributes = new HashMap<AttributeKey, Object>();
        attributes.put(AttributeKeys.FILL_COLOR, RelationalERDiagram.ENTITY_NAME_COMPARTMENT_COLOR);
        attributes.put(AttributeKeys.STROKE_COLOR, RelationalERDiagram.ENTITY_STROKE_COLOR);
        attributes.put(AttributeKeys.TEXT_COLOR, RelationalERDiagram.ENTITY_TEXT_COLOR);

        // Fundamental Entity
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEntityFundamental");
        tool = new CreationTool(new EntityFigure(Entity.TYPE_FUNDAMENTAL), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        // Associative Entity
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEntityAssociative");
        tool = new CreationTool(new EntityFigure(Entity.TYPE_ASSOCIATIVE), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        // Attributive Entity
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEntityAttributive");
        tool = new CreationTool(new EntityFigure(Entity.TYPE_ATTRIBUTIVE), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        // Supertype Entity
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEntitySupertype");
        tool = new CreationTool(new EntityFigure(Entity.TYPE_SUPERTYPE), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        // Subtype Entity
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEntitySubtype");
        tool = new CreationTool(new EntityFigure(Entity.TYPE_SUBTYPE), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        tb.addSeparator();

        // AttributeKeys for Relationships
        attributes = new HashMap<AttributeKey, Object>();
        attributes.put(AttributeKeys.FILL_COLOR, RelationalERDiagram.RELATION_FILL_COLOR);
        attributes.put(AttributeKeys.STROKE_COLOR, RelationalERDiagram.RELATION_STROKE_COLOR);

        // Relationship 1:1
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createRelationship11");
        rf = new RelationshipFigure();
        rf.getModel().setStartCardinality(1);
        rf.getModel().setEndCardinality(1);
        tool = new ConnectionTool(rf, attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        t = new JToggleButton();
        labels.configureToolBarButton(t, "createRelationshipM1");
        rf = new RelationshipFigure();
        rf.getModel().setStartCardinality(Relationship.MANY_CARDINALITY);
        rf.getModel().setEndCardinality(1);
        tool = new ConnectionTool(rf, attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        t = new JToggleButton();
        labels.configureToolBarButton(t, "createRelationship1M");
        rf = new RelationshipFigure();
        rf.getModel().setStartCardinality(1);
        rf.getModel().setEndCardinality(Relationship.MANY_CARDINALITY);
        tool = new ConnectionTool(rf, attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        t = new JToggleButton();
        labels.configureToolBarButton(t, "createRelationshipMM");
        rf = new RelationshipFigure();
        rf.getModel().setStartCardinality(Relationship.MANY_CARDINALITY);
        rf.getModel().setEndCardinality(Relationship.MANY_CARDINALITY);
        tool = new ConnectionTool(rf, attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        /*
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createRectangle");
        tool = new CreationTool(new RectangleFigure());
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEllipse");
        tool = new CreationTool(new EllipseFigure());
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        t = new JToggleButton();
        labels.configureToolBarButton(t, "createLine");
        tool = new CreationTool(new LineFigure());
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);

        AttributedFigure af;
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createArrow");
        tool = new CreationTool(af = new LineFigure());
        af.setAttribute(AttributedFigure.END_DECORATION, new ArrowTip(0.35, 12, 11.3));
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);


        t = new JToggleButton();
        labels.configureToolBarButton(t, "createLink");
        tool = new ConnectionTool(new LineConnection());
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
         */

        tb.addSeparator();

        t = new JToggleButton();
        drawLabels.configureToolBarButton(t, "createTextArea");
        tool = new TextAreaCreationTool(new TextAreaFigure());
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
    }

    /**
     * Handles toggle buttons for the tools.
     */
    private class ToolHandler extends ToolAdapter {

        private JToggleButton defaultToolButton;

        public ToolHandler(JToggleButton defaultToolButton) {
            this.defaultToolButton = defaultToolButton;
        }

        @Override
        public void toolDone(ToolEvent event) {
            defaultToolButton.setSelected(true);
        }
    }

    @Override
    public void markChangesAsSaved() {
        undo.discardAllEdits();
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        mainSplitPane = new javax.swing.JSplitPane();
        mainLeftPanel = new javax.swing.JPanel();
        optionsPanel = new ch.hslu.cm.OptionsPanel();
        mainRightPanel = new javax.swing.JPanel();
        mainPanel = new javax.swing.JPanel();
        tabbedPane = new javax.swing.JTabbedPane();
        problemScrollPane = new javax.swing.JScrollPane();
        problemDrawingView = new org.jhotdraw.draw.DefaultDrawingView();
        solutionScrollPane = new javax.swing.JScrollPane();
        solutionDrawingView = new org.jhotdraw.draw.DefaultDrawingView();

        setLayout(new java.awt.BorderLayout());

        mainLeftPanel.setLayout(new java.awt.BorderLayout());
        mainLeftPanel.add(optionsPanel, java.awt.BorderLayout.CENTER);

        mainSplitPane.setLeftComponent(mainLeftPanel);

        mainRightPanel.setMinimumSize(new java.awt.Dimension(0, 0));
        mainRightPanel.setLayout(new java.awt.BorderLayout());

        mainPanel.setLayout(new java.awt.BorderLayout());

        tabbedPane.setTabPlacement(javax.swing.JTabbedPane.BOTTOM);

        problemScrollPane.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        problemScrollPane.setViewportView(problemDrawingView);

        tabbedPane.addTab("null", problemScrollPane);

        solutionScrollPane.setBorder(javax.swing.BorderFactory.createEmptyBorder(0, 0, 0, 0));
        solutionScrollPane.setViewportView(solutionDrawingView);

        tabbedPane.addTab("null", solutionScrollPane);

        tabbedPane.setSelectedIndex(1);

        mainPanel.add(tabbedPane, java.awt.BorderLayout.CENTER);

        mainRightPanel.add(mainPanel, java.awt.BorderLayout.CENTER);

        mainSplitPane.setRightComponent(mainRightPanel);

        add(mainSplitPane, java.awt.BorderLayout.CENTER);
    }// </editor-fold>//GEN-END:initComponents
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel mainLeftPanel;
    private javax.swing.JPanel mainPanel;
    private javax.swing.JPanel mainRightPanel;
    private javax.swing.JSplitPane mainSplitPane;
    private ch.hslu.cm.OptionsPanel optionsPanel;
    private org.jhotdraw.draw.DefaultDrawingView problemDrawingView;
    private javax.swing.JScrollPane problemScrollPane;
    private org.jhotdraw.draw.DefaultDrawingView solutionDrawingView;
    private javax.swing.JScrollPane solutionScrollPane;
    private javax.swing.JTabbedPane tabbedPane;
    // End of variables declaration//GEN-END:variables
}
